/*
 This model is intended for saving newly created direct scheduling appointments.

 [POST] {base}/rest/direct-scheduling/site/{site-code}/patient/{assigning-authority}/{patient-id}/booked-appointments

 Unfortunately, the expectations of this endpoint is not well document and the data provided through
 this model, is base on what has historically been passed to the endpoint opposed to any specific design
 specification.  Hopefully, sometime in the future the exact API specification can be flushed out and
 this model can be updated accordingly.
 */
define([
    'underscore',
    'moment',
    'modules/appointments/appointments-module',
    'modules/new-appointment-request/resources/common/model',
], function(_, moment, module, Model) {
    'use strict';


    var AFTER_SUBMIT_SUCCESS_STATUS = 'Scheduled';
    var REGULAR_VISIT_TYPE = '9';
    var PATIENT_REQUEST = 'P';
    var DEFAULT_LEVEL = '1';
    var NONE = '';
    var DEFAULT_REQUEST_TYPE = 'NEXT_AVAILABLE_APPT';
    var DEFAULT_CARE_TYPE = 'REGULAR';
    var DEFAULT_APPT_KIND = 'TRADITIONAL';
    var DEFAULT_VIDEO_APPT_KIND = 'MOBILE_ANY';

    // TODO save events! To trigger out to the view.

    return Model.extend({
        resourceName: 'ds-booked-appointments',

        defaults: {
            apptType: PATIENT_REQUEST,
            purpose: REGULAR_VISIT_TYPE,
            ekg: NONE,
            lab: NONE,
            lvl: DEFAULT_LEVEL,
            xRay: NONE,
        },

        getUrl: function() {
            var resource = this.getResource();
            var siteCode = this.getSiteCode();
            var idType = this.getUserIdType();
            var id = this.getUserId();

            return resource.formatHref({
                'site-code': siteCode,
                'assigning-authority': idType,
                'patient-id': id,
            });
        },

        /**
         * Casts this model to an Appointments Model for the details page.
         * @return {Backbone.Model} appointment-module.appointments.model
         */
        toAppointmentsModel: function() {
            var link = this._createCancelLink();
            var clinicId = this.getClinicId();
            var clinicName = this.getClinicName();
            var clinic = this.clinic();
            var clinicFriendlyName = clinic ? clinic.getFriendlyLocationName() : '';
            var siteCode = this.getSiteCode();
            var facilityName = this.getFacilityName();
            var direct = this._directDataToAppointmentModel();
            var originalData = this.toJSON();

            var data = {
                link: link,
                currentStatus: AFTER_SUBMIT_SUCCESS_STATUS,
                siteCode: siteCode,
                facilityName: facilityName,
                clinic: {
                    id: clinicId,
                    friendlyName: clinicName,
                    clinicFriendlyName: clinicFriendlyName,
                },
                appointmentLength: direct.apptLength,
                appointmentTime: direct.dateTime,
            };

            var moduleData = _.extend({}, originalData, data, direct);
            moduleData.bookingNote = moduleData.bookingNotes;

            // eslint-disable-next-line new-cap
            return new module.appointments.model(moduleData);
        },

        createSaveData: function() {
            var direct = this._directSaveData();
            var defaults = _.extend({}, this.defaults);
            var patients = this._patientSaveData();
            var kind = this._appointmentTypeSaveData();
            var providers = this._providersSaveData();

            return _.extend({}, direct, defaults, patients, kind, providers);
        },

        /**
         * Creates information for the cancel link on the details page
         * @return {object}
         * @private
         */
        _createCancelLink: function() {
            var url = this.getUrl();
            var cancelUrl = url.replace('booked-appointments', 'cancel-appointment');

            return {
                href: cancelUrl,
                rel: 'related',
                title: 'cancel-appointment',
            };
        },

        /**
         * Transform data to appointment model
         * @returns {Object}
         * @private
         */
        _directDataToAppointmentModel: function() {
            var direct = this.get('direct');
            var data = _.pick(direct, [
                'apptLength',
                'dateTime',
                'desiredDate',
            ]);

            data.bookingNotes = direct.purpose;
            return data;
        },


        /**
         * Extracts the data from the direct object
         * @return {Object}
         * @private
         */
        _directSaveData: function() {
            var self = this;
            var direct = self.get('direct');

            return {
                desiredDate: self.isVideoVisit() && self.isTelehealthFeaturesEnabled ?
                    moment(direct.desiredDate).format() :
                    moment.utc(direct.desiredDate, 'MM/DD/YYYY HH:mm:ss').format(),
                dateTime: self.isVideoVisit() && self.isTelehealthFeaturesEnabled ?
                    moment(direct.dateTime).format() :
                    moment.utc(direct.dateTime, 'MM/DD/YYYY HH:mm:ss').format(),
                duration: direct.apptLength,
                bookingNotes: direct.purpose,
            };
        },

        /**
         * Extracts the patients identifer, name, contact information, and booking locations
         * @returns {{patients: {patient: Array}}}
         * @private
         */
        _patientSaveData: function() {
            var self = this;
            var user = self.user();
            var patients = {patients: {patient: []}};
            var patientData = {};

            patientData.id = {
                uniqueId: self.getUserId(),
                assigningAuthority: self.getUserIdType(),
            };

            patientData.name = {
                firstName: user.get('firstName'),
                lastName: user.get('lastName'),
                middleInitial: user.get('middleInitials'),
            };
            patientData.contactInformation = {
                preferredEmail: self.getEmailPreferences().get('emailAddress'),
                timeZone: self.getInstitutionTimezone(),
                // TODO placeholder until mobile data is found
                // mobile: this.emailPreferencesModel.get("phoneNumber")
            };
            patientData.location = {
                type: self.isVideoVisit() ? 'NonVA' : 'VA',
                facility: {
                    name: self.getFacilityName(),
                    siteCode: self.getSiteCode(),
                    timeZone: self.getInstitutionTimezone(),
                },
                clinic: {
                    ien: self.getClinicId(),
                    name: self.getClinicName(),
                },
            };
            patients.patients.patient.push(patientData);

            return patients;
        },

        /**
         * Extracts the scheduling request type, care type,
         * @returns {{schedulingRequestType: string, type: string, appointmentKind: string}}
         */
        _appointmentTypeSaveData: function() {
            var self = this;
            return {
                schedulingRequestType: DEFAULT_REQUEST_TYPE,
                type: DEFAULT_CARE_TYPE,
                appointmentKind: self.isVideoVisit() && self.isTelehealthFeaturesEnabled() ?
                    DEFAULT_VIDEO_APPT_KIND :
                    DEFAULT_APPT_KIND,
            };
        },

        /**
         * Extracts the providers location
         * @returns {{providers: {provider: Array}}}
         */
        _providersSaveData: function() {
            var self = this;
            var providers = {providers: {provider: []}};
            var providerData = {location: {type: 'VA'}};

            if (self.isVideoVisit() && self.isTelehealthFeaturesEnabled()) {
                providerData = {
                    contactInformation: {},
                    location: {
                        type: 'VA',
                        facility: {
                            name: self.getFacilityName(),
                            siteCode: self.getSiteCode(),
                            timeZone: self.getInstitutionTimezone(),
                        },
                        clinic: {
                            ien: self.getClinicId(),
                            name: self.getClinicName(),
                        },
                    },
                };
            }
            providers.providers.provider.push(providerData);

            return providers;
        },
    });
});
